const headers =  {
    'X-WP-Nonce': CF7Apps.nonce
};

/**
 * Fetches the menu items from the server.
 * 
 * @since 3.0.0
 */
export async function getMenu() {
    const response = await fetch(`${CF7Apps.restURL}cf7apps/v1/get-menu-items`, {
        headers: headers
    });

    if (!response.ok) {
        return false;
    }

    const json = await response.json();

    return json.data;
}

/**
 * Fetches the app settings from the server.
 * 
 * @param {string} id The ID of the app.
 * 
 * @since 3.0.0
 */
export async function getApps( id = '' ) {
    // Use REST route with optional /id param, e.g. .../get-apps or .../get-apps/123
    let url = `${CF7Apps.restURL}cf7apps/v1/get-apps`;
    if (id) {
        url += `/${encodeURIComponent(id)}`;
    }

    const response = await fetch(url, {
        headers: headers,
        method: 'GET'
    });

    if (!response.ok) {
        return false;
    }

    const json = await response.json();

    return json.data;
}

/**
 * Saves the App settings to the server.
 * 
 * @param {string} id The ID of the app. 
 * @param {object} app_settings The app settings to save.
 *  
 * @returns 
 * 
 * @since 3.0.0
 */
export async function saveSettings(id, app_settings) {
    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/save-app-settings`, {
            headers: headers,
            method: 'POST',
            body: JSON.stringify({ 
                id: id,
                app_settings
            })
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();

    return json.data;
}

/**
 * Fetches the CF7 forms from the server.
 *  
 * @since 3.0.0
 * 
 * @returns {array} The CF7 forms.
 */
export async function getCF7Forms() {
    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/get-cf7-forms`, {
            headers: headers,
            method: 'GET'
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();
    
    return json.data;
}

/**
 * If the app has migrated or not.
 * 
 * @returns {boolean} True if the app has migrated, false otherwise.
 * 
 * @since 3.0.0
 */
export async function hasMigrated() {
    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/has-migrated`, {
            headers: headers,
            method: 'GET'
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();
    
    return json.data;
}

/**
 * Migrates the app from old structure to new structure.
 * 
 * @returns {boolean} True if the migration was successful, false otherwise.
 * 
 * @since 3.0.0
 */
export async function migrate() {
    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/migrate`, {
            headers: headers,
            method: 'POST'
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();
    
    return json.data;
}

/**
 * Fetches the CF7 entries from the server.
 *
 * @since 3.1.0
 * @returns {Promise<*|boolean>}
 */
export async function getCF7Entries( { page = 1, perPage = 10, form_id = 0, search = '', start_date = 0, end_date = 0 } = {} ) {

    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/get-cf7-entries?page=${ page }&per-page=${ perPage }&form-id=${ form_id }&search=${ search }&start-date=${ start_date }&end-date=${ end_date }`, {
            headers: headers,
            method: 'GET'
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();

    return {
        entries: json.data,
        total: json.total,
    }
}

/**
 * Deletes the CF7 entries from the server.
 *
 * @since 3.1.0
 * @param entryIds Array of entry IDs to delete.
 *
 * @returns {Promise<*|boolean>}
 */
export async function deleteCF7Entries( entryIds = [] ) {
    if ( ! Array.isArray( entryIds ) || entryIds.length === 0 ) {
        return false;
    }

    // serialize the entry IDs to a query string
    const queryString = entryIds.map( id => `entry_ids[]=${ encodeURIComponent( id ) }` ).join( '&' );

    const response = await fetch( `${CF7Apps.restURL}cf7apps/v1/delete-cf7-entries?${ queryString }`, {
            headers: headers,
            method: 'GET',
    } );

    if ( ! response.ok ) {
        return false;
    }

    const json = await response.json();
    return json.data;
}

/**
 * Fetches all CF7 forms from the server.
 *
 * @since 3.1.0
 * @returns {Promise<*|boolean>}
 */
export async function getAllCF7Forms() {
    const response = await fetch(
        `${CF7Apps.restURL}cf7apps/v1/get-all-cf7-forms`, {
            headers: headers,
            method: 'GET'
        }
    );

    if (!response.ok) {
        return false;
    }

    const json = await response.json();

    return json.data;
}